<?php
namespace LiquidElementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Utils;
use Elementor\Repeater;
use Elementor\Group_Control_Image_Size;
use Elementor\Group_Control_Typography;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class LQD_Stack extends Widget_Base {

	public function get_name() {
		return 'lqd-stack';
	}

	public function get_title() {
		return __( 'Liquid Stack', 'logistics-hub-core' );
	}

	public function get_icon() {
		return 'eicon-parallax lqd-element';
	}

	public function get_categories() {
		return [ 'liquid-core' ];
	}

	public function get_keywords() {
		return [ 'scroll', 'stack', 'scrollable' ];
	}

	public function get_behavior() {
		$settings = $this->get_settings_for_display();
		$behavior = [];

		$behavior[] = [
			'behaviorClass' => 'LiquidGetElementComputedStylesBehavior',
			'options' => [
				'includeSelf' => 'true',
				'getRect' => 'true',
				'elementsSelector' => "'.lqd-stack-item'"
			]
		];

		$behavior[] = [
			'behaviorClass' => 'LiquidStackBehavior'
		];

		return $behavior;
	}

	protected function register_controls() {

		$this->start_controls_section(
			'general_section',
			[
				'label' => __( 'General', 'logistics-hub-core' ),
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'content_type',
			[
				'label' => __( 'Content Type', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'default' => 'tinymce',
				'options' => [
					'tinymce' => [
						'title' => __( 'TinyMCE', 'logistics-hub-core' ),
						'icon' => 'eicon-text-align-left'
					],
					'image' => [
						'title' => __( 'Image', 'logistics-hub-core' ),
						'icon' => 'eicon-image-bold'
					],
					'el_template' => [
						'title' => __( 'Elementor Template', 'logistics-hub-core' ),
						'icon' => 'eicon-site-identity'
					],
				],
			]
		);

		$repeater->add_control(
			'image',
			[
				'label' => esc_html__( 'Image', 'logistics-hub-core' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
				'condition' => [
					'content_type' => 'image'
				],
			]
		);

		$repeater->add_group_control(
			Group_Control_Image_Size::get_type(),
			[
				'name' => 'image',
				'default' => 'full',
				'separator' => 'after',
				'condition' => [
					'content_type' => 'image'
				],
			]
		);

		$repeater->add_control(
			'content_templates', [
				'label' => __( 'Select Template', 'logistics-hub-core' ),
				'type' => Controls_Manager::SELECT,
				'label_block' => true,
				'options' => liquid_helper()->get_elementor_templates(),
				'description' => liquid_helper()->get_elementor_templates_edit(),
				'default' => '0',
				'separator' => 'after',
				'condition' => [
					'content_type' => 'el_template'
				],
			]
		);

		$repeater->add_control(
			'item_content', [
				'label' => __( 'Content', 'logistics-hub-core' ),
				'type' => Controls_Manager::WYSIWYG,
				'separator' => 'after',
				'condition' => [
					'content_type' => 'tinymce'
				],
			]
		);

		\LQD_Elementor_Helper::add_style_controls(
			$repeater,
			'stack',
			[
				'item' => [
					'controls' => [
						[
							'type' => 'typography',
						],
						[
							'type' => 'padding',
						],
						[
							'type' => 'liquid_color',
						],
						[
							'type' => 'liquid_background_css',
						],
						[
							'type' => 'border',
						],
						[
							'type' => 'border_radius',
						],
						[
							'type' => 'box_shadow',
						],
					],
					'plural_heading' => false,
					'selector' => '{{CURRENT_ITEM}}',
					'state_tabs' => [ 'normal', 'hover', 'active' ],
					'state_selectors_before' => [ 'hover' => '{{CURRENT_ITEM}}', 'active' => '{{CURRENT_ITEM}}' ],
				]
			],
			true,
			true,
			true,
		);

		$this->add_control(
			'items',
			[
				'label' => __( 'Items', 'logistics-hub-core' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'content_type' => 'tinymce',
						'item_content' => __( 'Stackable item #1', 'logistics-hub-core' ),
					],
					[
						'content_type' => 'tinymce',
						'item_content' => __( 'Stackable item #2', 'logistics-hub-core' ),
					],
					[
						'content_type' => 'tinymce',
						'item_content' => __( 'Stackable item #3', 'logistics-hub-core' ),
					],
				],
				'title_field' => '{{{ content_type }}}',
				'separator' => 'after'
			]
		);

		$this->add_control(
			'loop',
			[
				'label' => esc_html__( 'Loop', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'adaptive_height',
			[
				'label' => esc_html__( 'Adaptive height', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'nav_buttons',
			[
				'label' => __( 'Navigation buttons', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'separator' => 'before'
			]
		);

		$this->add_control(
			'pagination_dots',
			[
				'label' => __( 'Pagination dots', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
			]
		);

		$this->add_control(
			'slides_indicator',
			[
				'label' => __( 'Slides indicator', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->end_controls_section();

		// Navigation Section
		$this->start_controls_section(
		'navigation_section',
			[
				'label' => __( 'Navigation', 'logistics-hub-core' ),
				'condition' => [
					'nav_buttons' => 'yes'
				]
			]
		);

		$this->add_control(
			'nav_arrows_style',
			[
				'label' => __( 'Style', 'logistics-hub-core' ),
				'type' => Controls_Manager::SELECT,
				'default' => '1',
				'options' => [
					'1' => __( 'Style 1', 'logistics-hub-core' ),
					'custom' => __( 'Custom', 'logistics-hub-core' ),
				],
			]
		);

		$this->add_control(
			'nav_prev_icon',
			[
				'label' => esc_html__( 'Previous icon', 'logistics-hub-core' ),
				'type' => Controls_Manager::ICONS,
				'skin' => 'inline',
				'default' => [
					'value' => 'fas fa-arrow-left',
					'library' => 'fa-solid',
				],
				'condition' => [
					'nav_arrows_style' => 'custom'
				]
			]
		);

		$this->add_control(
			'nav_next_icon',
			[
				'label' => esc_html__( 'Next icon', 'logistics-hub-core' ),
				'type' => Controls_Manager::ICONS,
				'skin' => 'inline',
				'default' => [
					'value' => 'fas fa-arrow-right',
					'library' => 'fa-solid',
				],
				'condition' => [
					'nav_arrows_style' => 'custom'
				]
			]
		);

		$this->add_responsive_control(
			'nav_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'outside' => [
						'title' => esc_html__( 'Outside', 'logistics-hub-core' ),
						'icon' => 'eicon-sign-out',
					],
					'inside' => [
						'title' => esc_html__( 'Inside', 'logistics-hub-core' ),
						'icon' => 'eicon-square',
					],
				],
				'default' => 'outside',
				'toggle' => false,
				'selectors_dictionary' => [
					'outside' => 'relative',
					'inside' => 'absolute'
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => 'position: {{VALUE}}'
				]
			]
		);

		$this->add_responsive_control(
			'nav_align',
			[
				'label' => __( 'Alignment', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-center',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
					'space-between' => [
						'title' => esc_html__( 'Space between', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-stretch',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => 'justify-content: {{VALUE}};'
				],
				'condition' => [
					'nav_separate_buttons_offset' => ''
				],
			]
		);

		$this->add_responsive_control(
			'nav_orientation_h',
			[
				'label' => esc_html__( 'Horizontal orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => '%',
					'size' => 0
				],
				'condition' => [
					'nav_orientation_h' => 'start'
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'nav_orientation_h' => 'end'
				],
			]
		);

		$this->add_responsive_control(
			'nav_orientation_v',
			[
				'label' => esc_html__( 'Vertical orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => 'top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'nav_orientation_v' => 'top'
				],
			]
		);

		$this->add_responsive_control(
			'nav_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'nav_orientation_v' => 'bottom'
				],
			]
		);

		$this->add_control(
			'nav_separate_buttons_offset',
			[
				'label' => __( 'Offset buttons separately', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
				]
			]
		);

		$this->add_responsive_control(
			'prev_button_orientation_h',
			[
				'label' => esc_html__( 'Prev button Horizontal orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-prev' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-prev' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => '%',
					'size' => 0
				],
				'condition' => [
					'prev_button_orientation_h' => 'start',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-prev' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'prev_button_orientation_h' => 'end',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_orientation_v',
			[
				'label' => esc_html__( 'Prev button vertical orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-prev' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-prev' => 'top: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => '%',
					'size' => 50
				],
				'condition' => [
					'prev_button_orientation_v' => 'top',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'prev_button_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-prev' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'prev_button_orientation_v' => 'bottom',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_orientation_h',
			[
				'label' => esc_html__( 'Next button horizontal orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'end',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-next' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-next' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => '%',
					'size' => 0
				],
				'condition' => [
					'next_button_orientation_h' => 'start',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-next' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => '%',
					'size' => 0
				],
				'condition' => [
					'next_button_orientation_h' => 'end',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_orientation_v',
			[
				'label' => esc_html__( 'Next button vertical orientation', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-next' => '{{VALUE}}',
				],
				'condition' => [
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-next' => 'top: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => '%',
					'size' => 50
				],
				'condition' => [
					'next_button_orientation_v' => 'top',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_responsive_control(
			'next_button_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh', 'custom' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-nav-next' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'default' => [
					'unit' => '%',
					'size' => 50
				],
				'condition' => [
					'next_button_orientation_v' => 'bottom',
					'nav_separate_buttons_offset' => 'yes'
				],
			]
		);

		$this->add_control(
			'nav_appear_onhover',
			[
				'label' => __( 'Appear on hover', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
					'nav_placement' => 'inside'
				]
			]
		);

		$this->end_controls_section();

		// Pagination Dots
		$this->start_controls_section(
			'pagination_dots_section',
			[
				'label' => __( 'Pagination dots', 'logistics-hub-core' ),
				'condition' => [
					'pagination_dots' => 'yes'
				]
			]
		);

		$this->add_control(
			'dots_style',
			[
				'label' => __( 'Style', 'logistics-hub-core' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'1' => __( 'Style 1', 'logistics-hub-core' ),
				],
				'default' => '1'
			]
		);

		$this->add_responsive_control(
			'dots_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'outside' => [
						'title' => esc_html__( 'Outside', 'logistics-hub-core' ),
						'icon' => 'eicon-sign-out',
					],
					'inside' => [
						'title' => esc_html__( 'Inside', 'logistics-hub-core' ),
						'icon' => 'eicon-square',
					],
				],
				'default' => 'outside',
				'toggle' => false,
				'selectors_dictionary' => [
					'outside' => 'relative',
					'inside' => 'absolute',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => 'position: {{VALUE}};'
				]
			]
		);

		$this->add_responsive_control(
			'dots_align',
			[
				'label' => __( 'Alignment', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-center',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
					'space-between' => [
						'title' => esc_html__( 'Space between', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-stretch',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => 'justify-content: {{VALUE}};'
				],
				'condition' => [
					'dots_placement' => 'outside'
				]
			]
		);

		$this->add_responsive_control(
			'dots_placement_h',
			[
				'label' => esc_html__( 'Horizontal placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'selectors_dictionary' => [
					'start' => 'inset-inline-end: auto',
					'end' => 'inset-inline-start: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_h' => 'start'
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_h' => 'end'
				],
			]
		);

		$this->add_responsive_control(
			'dots_placement_v',
			[
				'label' => esc_html__( 'Vertical placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'selectors_dictionary' => [
					'top' => 'bottom: auto',
					'bottom' => 'top: auto',
				],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => '{{VALUE}}',
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => 'top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_v' => 'top'
				],
			]
		);

		$this->add_responsive_control(
			'dots_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-dots' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'dots_placement_v' => 'bottom'
				],
			]
		);

		$this->end_controls_section();

		// Slides Indicator Section
		$this->start_controls_section(
			'slides_indicator_section',
			[
				'label' => __( 'Slides indicator', 'logistics-hub-core' ),
				'condition' => [
					'slides_indicator' => 'yes'
				]
			]
		);

		$this->add_control(
			'slides_indicator_inline_w_nav',
			[
				'label' => __( 'Inline with navigation?', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
					'slides_indicator' => 'yes',
					'nav_buttons' => 'yes',
					'slides_indicator_inline_w_dots' => '',
				]
			]
		);

		$this->add_responsive_control(
			'slides_indicator_inline_w_nav_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'center' => [
						'title' => esc_html__( 'Center', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-center',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'condition' => [
					'slides_indicator' => 'yes',
					'nav_buttons' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => 'yes',
				]
			]
		);

		$this->add_control(
			'slides_indicator_inline_w_dots',
			[
				'label' => __( 'Inline with dots?', 'logistics-hub-core' ),
				'type' => Controls_Manager::SWITCHER,
				'condition' => [
					'slides_indicator' => 'yes',
					'pagination_dots' => 'yes',
					'slides_indicator_inline_w_nav' => '',
				]
			]
		);

		$this->add_responsive_control(
			'slides_indicator_inline_w_dots_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'default' => 'center',
				'toggle' => false,
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => 'yes',
					'slides_indicator_inline_w_nav' => '',
				]
			]
		);

		$this->add_control(
			'slides_indicator_placement',
			[
				'label' => __( 'Placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					// 'outside' => [
					// 	'title' => esc_html__( 'Outside', 'logistics-hub-core' ),
					// 	'icon' => 'eicon-sign-out',
					// ],
					'inside' => [
						'title' => esc_html__( 'Inside', 'logistics-hub-core' ),
						'icon' => 'eicon-square',
					],
				],
				'default' => 'inside',
				'toggle' => false,
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
				]
			]
		);

		$this->add_control(
			'slides_indicator_placement_h',
			[
				'label' => esc_html__( 'Horizontal placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'start' => [
						'title' => esc_html__( 'Start', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-left',
					],
					'end' => [
						'title' => esc_html__( 'End', 'logistics-hub-core' ),
						'icon' => 'eicon-h-align-right',
					],
				],
				'toggle' => false,
				'default' => 'start',
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_x',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-slides-ind' => 'inset-inline-start: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_h' => 'start'
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_x_end',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vw' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-slides-ind' => 'inset-inline-end: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_h' => 'end'
				],
			]
		);

		$this->add_control(
			'slides_indicator_placement_v',
			[
				'label' => esc_html__( 'Vertical placement', 'logistics-hub-core' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'top' => [
						'title' => esc_html__( 'Top', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-top',
					],
					'bottom' => [
						'title' => esc_html__( 'Bottom', 'logistics-hub-core' ),
						'icon' => 'eicon-v-align-bottom',
					],
				],
				'toggle' => false,
				'default' => 'top',
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_y',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-slides-ind' => 'top: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_v' => 'top'
				],
			]
		);

		$this->add_responsive_control(
			'slides_indicator_offset_y_bottom',
			[
				'label' => esc_html__( 'Offset', 'logistics-hub-core' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%', 'vh' ],
				'selectors' => [
					'{{WRAPPER}} .lqd-stack-slides-ind' => 'bottom: {{SIZE}}{{UNIT}}',
				],
				'condition' => [
					'slides_indicator' => 'yes',
					'slides_indicator_inline_w_dots' => '',
					'slides_indicator_inline_w_nav' => '',
					'slides_indicator_placement' => 'inside',
					'slides_indicator_placement_v' => 'bottom'
				],
			]
		);

		$this->end_controls_section();

		\LQD_Elementor_Helper::add_style_controls(
			$this,
			'stack',
			[
				'stack' => [
					'label' => 'General',
					'controls' => [
						[
							'type' => 'height',
							'condition' => [
								'adaptive_height' => ''
							]
						],
					],
					'plural_heading' => false,
					'state_tabs' => [ 'normal', 'hover' ],
					'state_selectors_before' => [ 'hover' => '{{WRAPPER}}' ]
				],
				'nav' => [
					'label' => 'Navigation button',
					'controls' => [
						[
							'type' => 'font_size',
							'label' => 'Icon size',
							'css_var' => '--lqd-stack-nav-icon-size'
						],
						[
							'type' => 'gap',
							'label' => 'Gap between buttons',
							'css_var' => '--lqd-stack-nav-gap'
						],
						[
							'type' => 'liquid_linked_dimensions',
							'css_var' => '--lqd-stack-nav-btn'
						],
						[
							'type' => 'liquid_color',
							'css_var' => '--lqd-stack-nav-btn-color',
						],
						[
							'type' => 'liquid_background_css',
							'css_var' => '--lqd-stack-nav-btn-bg',
						],
						[
							'type' => 'border',
							'css_var' => '--lqd-stack-nav-btn-br'
						],
						[
							'type' => 'border_radius',
							'css_var' => '--lqd-stack-nav-btn-brr'
						],
						[
							'type' => 'box_shadow',
							'css_var' => '--lqd-stack-nav-btn-bs'
						],
						[
							'type' => 'raw',
							'raw_options' => [
								'nav_rotate',
								[
									'label' => esc_html__( 'Rotation', 'logistics-hub-core' ),
									'type' => Controls_Manager::SLIDER,
									'range' => [
										'px' => [
											'min' => -360,
											'max' => 360,
											'step' => 1,
										],
									],
									'selectors' => [
										'{{WRAPPER}} .lqd-stack-nav' => 'transform: rotate({{SIZE}}deg)',
									],
								]
							],
							'tab' => 'none',
							'responsive' => true
						],
					],
					'condition' => [
						'nav_buttons' => 'yes'
					],
					'state_tabs' => [ 'normal', 'hover' ],
					'state_selectors' => [ 'hover' => ' .lqd-stack-nav-btn:not([disabled]):hover' ]
				],
				'dots' => [
					'label' => 'Pagination dot',
					'controls' => [
						[
							'type' => 'gap',
							'label' => 'Gap between dots',
							'css_var' => '--lqd-stack-dots-gap'
						],
						[
							'type' => 'liquid_linked_dimensions',
							'css_var' => '--lqd-stack-dots'
						],
						[
							'type' => 'margin',
							'css_var' => '--lqd-stack-dots-m'
						],
						[
							'type' => 'liquid_background_css',
							'css_var' => '--lqd-stack-dots-bg',
						],
						[
							'type' => 'border',
							'css_var' => '--lqd-stack-dots-br'
						],
						[
							'type' => 'border_radius',
							'css_var' => '--lqd-stack-dots-brr',
							'selector' => '.lqd-stack-dots .lqd-stack-dot'
						],
						[
							'type' => 'box_shadow',
							'css_var' => '--lqd-stack-dots-bs'
						],
						[
							'type' => 'raw',
							'raw_options' => [
								'dots_rotate',
								[
									'label' => esc_html__( 'Rotation', 'logistics-hub-core' ),
									'type' => Controls_Manager::SLIDER,
									'range' => [
										'px' => [
											'min' => -360,
											'max' => 360,
											'step' => 1,
										],
									],
									'selectors' => [
										'{{WRAPPER}} .lqd-stack-dots' => 'transform: rotate({{SIZE}}deg)',
									],
								]
							],
							'tab' => 'none',
							'responsive' => true
						],
					],
					'condition' => [
						'pagination_dots' => 'yes'
					],
					'state_tabs' => [ 'normal', 'hover', 'active' ],
					'state_selectors' => [ 'hover' => ' .lqd-stack-dot:hover', 'active' => ' .lqd-stack-dot-active' ]
				],
				'slides_indicator' => [
					'controls' => [
						[
							'type' => 'typography',
						],
						[
							'type' => 'gap',
						],
						[
							'type' => 'width',
							'label' => 'Indicator width',
							'css_var' => '--lqd-stack-slide-progress-w'
						],
						[
							'type' => 'height',
							'label' => 'Indicator height',
							'css_var' => '--lqd-stack-slide-progress-h'
						],
						[
							'type' => 'liquid_background_css',
							'name' => 'slides_indicator_bar_wrap_bg',
							'label' => 'Indicator background',
							'css_var' => '--lqd-stack-slide-bar-wrap-bg',
							'fields_options' => [
								'background' => [
									'label' => esc_html__( 'Bar background', 'logistics-hub-core' ),
								],
							],
						],
						[
							'type' => 'liquid_background_css',
							'name' => 'slides_indicator_bar_bg',
							'label' => 'Indicator bar background',
							'css_var' => '--lqd-stack-slide-bar-bg',
							'fields_options' => [
								'background' => [
									'label' => esc_html__( 'Filled bar background', 'logistics-hub-core' ),
								],
							],
						],
						[
							'type' => 'liquid_color',
							'name' => 'slides_indicator_color',
							'label' => 'Numbers color'
						],
					],
					'selector' => '.lqd-stack-slides-ind',
					'condition' => [
						'slides_indicator' => 'yes'
					]
				]
			],
		);

	}

	protected function get_nav_icons_style_custom( $settings ) {
		$arrows = [
			'prev' => '',
			'next' => ''
		];

		if ( !empty( $settings['nav_prev_icon']['value'] ) ) {
			$arrows['prev'] = Icons_Manager::try_get_icon_html( $settings['nav_prev_icon'], [ 'aria-hidden' => 'true', 'class' => 'w-1em h-auto align-middle fill-current relative' ] );
		}
		if ( !empty( $settings['nav_next_icon']['value'] ) ) {
			$arrows['next'] = Icons_Manager::try_get_icon_html( $settings['nav_next_icon'], [ 'aria-hidden' => 'true', 'class' => 'w-1em h-auto align-middle fill-current relative' ] );
		}

		return $arrows;
	}

	protected function get_nav_icons_style_1( $settings ) {
		return [
			'prev' => '<svg width="16" height="13" viewBox="0 0 16 13" fill="currentColor" xmlns="http://www.w3.org/2000/svg"><path d="M6.029 -9.53674e-07L6.972 0.942999L2.522 5.393L15.688 5.393V6.706L2.522 6.706L6.972 11.156L6.029 12.099L0.451004 6.525L3.8147e-06 6.053L0.451004 5.581L6.029 -9.53674e-07Z"/></svg>',
			'next' => '<svg width="16" height="13" viewBox="0 0 16 13" fill="currentColor" xmlns="http://www.w3.org/2000/svg"><path d="M9.659 0L8.716 0.943L13.166 5.393H0V6.706H13.166L8.716 11.156L9.659 12.099L15.237 6.525L15.688 6.053L15.237 5.581L9.659 0Z"/></svg>'
		];
	}

	protected function get_nav( $settings ) {

		if ( empty( $settings['nav_buttons'] ) ) return '';

		$nav_classnames = [ 'lqd-stack-nav', 'flex', 'items-center', 'w-full', 'pointer-events-none' ];
		$buttons_common_classnames = [ 'lqd-stack-nav-btn', 'inline-flex', 'items-center', 'justify-center', 'pointer-events-auto', 'transition-all', 'cursor-pointer', 'z-2' ];
		$nav_style = $settings['nav_arrows_style'];
		$slides_indicator = $settings['slides_indicator'];
		$slides_indicator_is_inline = $settings['slides_indicator_inline_w_nav'];
		$slides_indicator_placement = $settings['slides_indicator_inline_w_nav_placement'];
		$should_get_slides_indicator = $slides_indicator === 'yes' && $slides_indicator_is_inline === 'yes';

		if ( $settings['nav_separate_buttons_offset'] === 'yes' ) {
			$buttons_common_classnames[] = 'absolute';
		} else {
			$buttons_common_classnames[] = 'relative';
		}
		if ( $settings['nav_appear_onhover'] === 'yes' ) {
			$buttons_common_classnames[] = 'opacity-0';
			$buttons_common_classnames[] = 'lqd-group-carousel-hover:opacity-100';
		}

		$this->add_render_attribute( 'carousel_nav_prev_btn', [
			'class' => array_merge(
				['lqd-stack-nav-prev'],
				$buttons_common_classnames
			),
			'type' => 'button',
			'aria-label' => esc_attr__('Previous', 'logistics-hub-core'),
		] );
		$this->add_render_attribute( 'carousel_nav_next_btn', [
			'class' => array_merge(
				['lqd-stack-nav-next'],
				$buttons_common_classnames
			),
			'type' => 'button',
			'aria-label' => esc_attr__('Next', 'logistics-hub-core'),
		] );
		$this->add_render_attribute( 'carousel_nav', [
			'class' => $nav_classnames
		] );

		$nav_icons = $this->{'get_nav_icons_style_' . $nav_style}( $settings );

		?><div <?php $this->print_render_attribute_string( 'carousel_nav' ) ?>>
			<?php if ( $should_get_slides_indicator && $slides_indicator_placement === 'start' ) {
				$this->get_slides_indicator( $settings );
			} ?>
			<button <?php $this->print_render_attribute_string('carousel_nav_prev_btn'); ?>><?php
				echo $nav_icons['prev'];
			?></button>
			<?php if ( $should_get_slides_indicator && $slides_indicator_placement === 'center' ) {
				$this->get_slides_indicator( $settings );
			} ?>
			<button <?php $this->print_render_attribute_string('carousel_nav_next_btn'); ?>><?php
				echo $nav_icons['next'];
			?></button>
			<?php if ( $should_get_slides_indicator && $slides_indicator_placement === 'end' ) {
				$this->get_slides_indicator( $settings );
			} ?>
		</div><?php

	}

	protected function get_dots( $settings ) {

		if ( empty( $settings['pagination_dots'] ) ) return '';

		$items_count = count( $settings['items'] );
		$i = 1;

		?>

		<div class="lqd-stack-dots flex relative z-2"><?php
			while ($i <= $items_count) {
				?><span class="lqd-stack-dot transition-all cursor-pointer <?php echo $i === 1 ? 'lqd-stack-dot-active' : '' ?>"></span><?php
				$i++;
			}
		?></div>

		<?php

	}

	protected function get_slides_indicator( $settings ) {

		if ( empty( $settings['slides_indicator'] ) ) return '';

		$items_count = count( $settings['items'] );
		$i = 1;

		$this->add_render_attribute(
			'slides-indicator-wrapper',
			'class',
			'lqd-stack-slides-ind flex flex-col items-center gap-20 absolute top-1/2 end-0 z-2 lqd-transform -translate-y-1/2 text-20 font-bold text-black lqd-stack-slides-ind-'. esc_attr( $this->get_id() )
		);

		?><div <?php $this->print_render_attribute_string( 'slides-indicator-wrapper') ?>>
			<span class="lqd-stack-slides-ind-progress-wrap flex items-center relative">
				<span class="lqd-stack-slides-ind-progress-bg inline-block w-full h-full absolute top-0 start-0 lqd-transform translate-y-0 origin-center-top transition-all"></span>
				<span class="lqd-stack-slides-ind-progress-bg lqd-stack-slides-ind-progress-bar inline-block w-full h-full absolute top-0 start-0 lqd-transform translate-y-0 origin-center-top transition-all"></span>
			</span>
			<span class="lqd-stack-slides-ind-current shrink-0 relative overflow-hidden h-1em leading-none">
				<span class="lqd-stack-slides-ind-current-slider shrink-0 flex flex-col h-1em leading-none lqd-transform transition-transform"><?php
					while ($i <= $items_count) {
						?><span class="lqd-stack-slides-ind-current-num inline-flex items-center h-1em leading-none"><?php echo $i ?></span><?php
						$i++;
					}
				?></span>
			</span>
			<svg class="h-1em w-auto" width="12" height="14" viewBox="0 -2 14 16" fill="none" xmlns="http://www.w3.org/2000/svg">
				<path d="M0.324017 13.3749L9.51044 0.572428H11.2784L2.10502 13.3749H0.324017Z" fill="currentColor"/>
			</svg>
			<span class="lqd-stack-slides-ind-total shrink-0 flex items-center h-1em leading-none"><?php echo $items_count ?></span>
			<span class="lqd-stack-slides-ind-progress-wrap flex items-center relative">
				<span class="lqd-stack-slides-ind-progress-bg inline-block w-full h-full absolute top-0 start-0 lqd-transform translate-y-0 origin-center-top transition-all"></span>
				<span class="lqd-stack-slides-ind-progress-bg lqd-stack-slides-ind-progress-bar inline-block w-full h-full absolute bottom-0 start-0 lqd-transform translate-y-0 origin-center-bottom transition-all"></span>
			</span>
		</div><?php

	}

	protected function get_content_type_tinymce( $item ) {

		echo $item['item_content'];

	}

	protected function get_content_type_image( $item ) {

		?>
		<figure class="lqd-stack-content-img overflow-hidden"><?php
			Group_Control_Image_Size::print_attachment_image_html( $item );
		?></figure>
		<?php

	}

	protected function get_content_type_el_template( $item ) {

		echo \Elementor\Plugin::instance()->frontend->get_builder_content( $item[ 'content_templates' ], $with_css = true );

	}

	protected function get_content( $item ) {

		/**
		 * @type {string} tinymce | image | el_template
		 */
		$content_type = $item['content_type'];

		$this->{'get_content_type_' . $content_type}( $item );

	}

	protected function get_contents( $settings ) {

		$adaptive_height_enabled = $settings['adaptive_height'] === 'yes';
		$items = $settings['items'];
		$content_common_classnames = [ 'lqd-stack-item', 'w-full', 'top-0', 'start-0', 'overflow-y-auto' ];

		foreach ( $items as $i => $item ) {

			$content_attrs_id = $this->get_repeater_setting_key( 'content', 'items', $i );
			$content_classnames = $content_common_classnames;

			$content_classnames[] = 'elementor-repeater-item-' . esc_attr( $item['_id'] ) . '';

			if ( $i === 0 ) {
				$content_classnames[] = 'lqd-is-active z-1';
			} else {
				$content_classnames[] = 'opacity-0 invisible pointer-events-none z-0';
			}

			if ( $adaptive_height_enabled ) {
				if ( $i === 0 ) {
					$content_classnames[] = 'relative';
				} else {
					$content_classnames[] = 'absolute';
				}
			} else {
				$content_classnames[] = 'h-full absolute';
			}

			$this->add_render_attribute( $content_attrs_id, [
				'class' => $content_classnames,
			] );

		?>

		<div <?php $this->print_render_attribute_string( $content_attrs_id ) ?>>
			<div class="lqd-stack-item-inner"><?php $this->get_content( $item ); ?></div>
		</div>

		<?php }

	}

	protected function render() {

		$settings = $this->get_settings_for_display();
		$adaptive_height = $settings['adaptive_height'];

		$this->add_render_attribute( 'wrap', [
			'class' => 'lqd-stack-stack relative overflow-hidden'
		] );

		if ( $adaptive_height !== 'yes' ) {
			$this->add_render_attribute( 'wrap', 'class', 'h-svh' );
		}

        ?>

		<div <?php $this->print_render_attribute_string('wrap') ?>><?php
			$this->get_contents( $settings );
		?></div>

		<?php
			$this->get_nav( $settings );
			$this->get_dots( $settings );
			$this->get_slides_indicator( $settings );
		?>

        <?php

	}

}
\Elementor\Plugin::instance()->widgets_manager->register( new LQD_Stack() );